<?php

namespace Adfox\Packages\Livewire;

use App\Models\PackageItem;
use App\Settings\MollieSettings;
use App\Settings\PaystackSettings;
use App\Settings\RazorpaySettings;
use Filament\Forms\Components\Radio;
use Livewire\Component;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use App\Settings\StripeSettings;
use App\Settings\PaypalSettings;
use App\Settings\PaymentSettings;
use App\Settings\FlutterwaveSettings;
use App\Settings\OfflinePaymentSettings;
use App\Settings\PaymongoSettings;
use Carbon\Carbon;
use Akaunting\Money\Currency as AkauntingCurrency;
use App\Settings\PayuSettings;
use App\Settings\PhonePeSettings;

class PackagePayment extends Component implements HasForms
{
    use InteractsWithForms;

    public $currentPayment;
    public $payment_method;
    public $packageItems;
    public $type = 'PKG';
    public $ad_id;
    public $pkg_type;
    public $packageItemIds;
    public $promotions;
    public $subtotal = 0;
    public $tax = 0; // Define tax rate if applicable
    public $total = 0;
    public $isDifferentRate = false;
    public $convertedTotal = 0;
    public $defaultCurrency;
    public $selectedPaymentMethod;
    public $selectedPaymentInstruction;

    protected $paymentGateways = [
        'stripe' => 'payment.stripe-payment',
        'paypal' => 'paypal-payment',
        'flutterwave' => 'flutterwave-payment',
        'offline' => 'offline-payment',
        'paymongo' => 'paymongo-payment',
        'razorpay' => 'razorpay-payment',
        'paystack' => 'paystack-payment',
        'phonepe' => 'phonepe-payment',
        'payu' => 'payu-payment',
        'mollie' => 'mollie-payment',

    ];
    public $id; // Ad ID property

    /**
     * Initialize the component with the selected package items and set up payment options.
     *
     * @param array $selectedItems Array of selected item IDs.
     */
    public function mount($selectedItems)
    {
        $this->packageItemIds = $selectedItems;
        $this->initializePaymentOptions();
        $this->updatePaymentData();
    }

    /**
     * Update the payment-related data based on selected package items.
     */
    protected function updatePaymentData()
    {
        $this->packageItems = PackageItem::whereIn('id', $this->packageItemIds)->get();

        $this->subtotal = 0;
        $today = Carbon::today();

        foreach ($this->packageItemIds as $itemId) {
            $item = PackageItem::find($itemId);
            if ($item) {
                // Check if there is an offer price and it's valid for today
                $price = $item->price;
                if ($item->offer_enabled && $item->offer_price && $today->between($item->offer_start, $item->offer_end)) {
                    $price = $item->offer_price;
                }

                $this->subtotal += $price;
            }
        }
        // Accessing PaymentSettings
        $paymentSettings = app(PaymentSettings::class);

        if ($paymentSettings->enable_tax) {
            if ($paymentSettings->tax_type === 'percentage') {
                // Calculate tax as a percentage of the subtotal
                $this->tax = ($this->subtotal * $paymentSettings->tax_rate) / 100;
            } else if ($paymentSettings->tax_type === 'fixed') {
                // Apply a fixed tax rate
                $this->tax = $paymentSettings->tax_rate;
            }
        } else {
            // No tax applied
            $this->tax = 0;
        }

        // Add tax calculation logic here if necessary
        $this->total = $this->subtotal + $this->tax;


        $this->defaultCurrency =  $paymentSettings->currency;
        $paymentGatewayRate = $this->getPaymentGatewayRate();
        $systemExchangeRate = app(PaymentSettings::class)->exchange_rate;

        $this->isDifferentRate = $paymentGatewayRate != 1.0 && $paymentGatewayRate != $systemExchangeRate;
        $this->convertedTotal = $this->total * $paymentGatewayRate / $systemExchangeRate;
    }

    /**
     * Retrieve the current payment gateway exchange rate.
     *
     * @return float The exchange rate of the selected payment gateway.
     */
    private function getPaymentGatewayRate()
    {
        return match ($this->payment_method) {
            'stripe' => app(StripeSettings::class)->exchange_rate,
            'paypal' => app(PaypalSettings::class)->exchange_rate,
            'flutterwave' => app(FlutterwaveSettings::class)->exchange_rate,
            'offline' =>  app(OfflinePaymentSettings::class)->exchange_rate,
            'paymongo' => app(PaymongoSettings::class)->exchange_rate,
            'razorpay' => app(RazorpaySettings::class)->exchange_rate,
            'paystack' => app(PaystackSettings::class)->exchange_rate,
            'phonepe' => app(PhonePeSettings::class)->exchange_rate,
            'payu' => app(PayuSettings::class)->exchange_rate,
            'mollie' => app(MollieSettings::class)->exchange_rate,
            default => 1.0
        };
    }

    public function getExchangeCurrencySymbol()
    {
        return match ($this->payment_method) {
            'stripe' => (new AkauntingCurrency(app(StripeSettings::class)->currency))->getSymbol(),
            'paypal' => (new AkauntingCurrency(app(PaypalSettings::class)->currency))->getSymbol(),
            'flutterwave' => (new AkauntingCurrency(app(FlutterwaveSettings::class)->currency))->getSymbol(),
            'offline' => (new AkauntingCurrency(app(OfflinePaymentSettings::class)->currency))->getSymbol(),
            'paymongo' => (new AkauntingCurrency(app(PaymongoSettings::class)->currency))->getSymbol(),
            'razorpay' => (new AkauntingCurrency(app(RazorpaySettings::class)->currency))->getSymbol(),
            'paystack' => (new AkauntingCurrency(app(PaystackSettings::class)->currency))->getSymbol(),
            'phonepe' => (new AkauntingCurrency(app(PhonePeSettings::class)->currency))->getSymbol(),
            // 'payu' => (new AkauntingCurrency(app(PayuSettings::class)->currency))->getSymbol(),
            'mollie' => (new AkauntingCurrency(app(MollieSettings::class)->currency))->getSymbol(),
            default => '$'
        };
    }

    /**
     * Initialize available payment options based on system settings.
     *
     * @return array Array of available payment options.
     */
    protected function initializePaymentOptions()
    {
        $paymentOptions = [];

        if (app(StripeSettings::class)->status) {
            $paymentOptions['stripe'] = app(StripeSettings::class)->name;
        }

        if (app('filament')->hasPlugin('paypal') && app(PaypalSettings::class)->status) {
            $paymentOptions['paypal'] = app(PaypalSettings::class)->name;
        }

        if (app('filament')->hasPlugin('flutterwave') && app(FlutterwaveSettings::class)->status) {
            $paymentOptions['flutterwave'] = app(FlutterwaveSettings::class)->name;
        }

        if (app('filament')->hasPlugin('offline-payment') && app(OfflinePaymentSettings::class)->status) {
            // $paymentOptions['offline'] = app(OfflinePaymentSettings::class)->name;
            $offlineSettings = app(OfflinePaymentSettings::class);

            if (!empty($offlineSettings->payment_type)) {
                foreach ($offlineSettings->payment_type as $type) {
                    $key = 'offline_' . $type['name'];
                    $paymentOptions[$key] = $type['name'];
                    $this->paymentGateways[$key] = 'offline-payment'; // Add to paymentGateways
                }
            }
        }

        if (app('filament')->hasPlugin('paymongo') && app(PaymongoSettings::class)->status) {
            $paymentOptions['paymongo'] = app(PaymongoSettings::class)->name;
        }

        if (app('filament')->hasPlugin('razorpay') && app(RazorpaySettings::class)->status) {
            $paymentOptions['razorpay'] = app(RazorpaySettings::class)->name;
        }

        if (app('filament')->hasPlugin('paystack') && app(PaystackSettings::class)->status) {
            $paymentOptions['paystack'] = app(PaystackSettings::class)->name;
        }

        if (app('filament')->hasPlugin('phonepe') && app(PhonePeSettings::class)->status) {
            $paymentOptions['phonepe'] = app(PhonePeSettings::class)->name;
        }

        if (app('filament')->hasPlugin('payu') && app(PayuSettings::class)->status) {
            $paymentOptions['payu'] = app(PayuSettings::class)->name;
        }

        if (app('filament')->hasPlugin('mollie') && app(MollieSettings::class)->status) {
            $paymentOptions['mollie'] = app(MollieSettings::class)->name;
        }

        // Set default payment method if only one option is enabled
        if (count($paymentOptions) === 1) {
            $defaultMethod = array_key_first($paymentOptions);
            $this->payment_method = $defaultMethod;
            $this->selectedPaymentMethod = $defaultMethod;
            $this->currentPayment = $this->paymentGateways[$defaultMethod];
        }

        return $paymentOptions;
    }

    /**
     * Method to get payment data in the specified format.
     *
     * @return array
     */
    public function getPaymentDataProperty()
    {
        return [
            'user_id' => auth()->id(),
            'packageItemIds' => $this->packageItemIds,
            'subtotal' => $this->subtotal,
            'ad_id' => $this->ad_id,
            'action_type' => $this->pkg_type == 'single' ? 'ad-upgrade' : 'package-purchase',
            'tax' => $this->tax,
            'total' => $this->total
        ];
    }

    /**
     * Define the form schema for payment method selection.
     *
     * @param Form $form The filament form instance.
     * @return Form The configured form schema.
     */
    public function form(Form $form): Form
    {
        $paymentOptions = $this->initializePaymentOptions();

        return $form
            ->schema([
                Radio::make('payment_method')
                    ->hiddenLabel()
                    ->live()
                    ->options($paymentOptions)
                    ->afterStateUpdated(function ($state) {
                        if (app('filament')->hasPlugin('offline-payment') && app(OfflinePaymentSettings::class)->status) {
                            // Reset currentPayment before selecting a new one
                            $this->currentPayment = null;
                            $this->selectedPaymentMethod = str_replace('offline_', '', $state);

                            // Filter the payment types and find the matching one
                            $filtered = array_filter(app(OfflinePaymentSettings::class)->payment_type, function ($item) {
                                return $item['name'] == $this->selectedPaymentMethod;
                            });

                            // If a match is found, update the selected payment instruction
                            foreach ($filtered as $item) {
                                $this->selectedPaymentInstruction = $item['instruction'];
                            }
                            // Set current payment method
                            $this->dispatch('post-created', name: $this->selectedPaymentMethod, instruction: $this->selectedPaymentInstruction);
                        }
                        $this->currentPayment = $this->paymentGateways[$state] ?? null;
                        // Update any associated payment data
                        $this->updatePaymentData();
                    }),
            ]);
    }

    /**
     * Render the component view.
     *
     * @return \Illuminate\View\View The view to render.
     */
    public function render()
    {
        return view('packages::package-payment');
    }
}
