<?php

namespace Adfox\Phonepe\Livewire;

use App\Models\{WebhookPackage, WebhookUpgrade};
use App\Traits\{HandlesPackageApplication, HandlesPromotionApplication};
use Livewire\Component;
use App\Settings\{PaymentSettings, GeneralSettings, PhonePeSettings};
use PhonePe\PhonePe;

class PhonepePayment extends Component
{
    use HandlesPackageApplication, HandlesPromotionApplication;

    public $id;
    public $type;
    public $total;
    public $data;
    public $tax;
    public $subtotal;
    public $successUrl;
    public $payment_gateway_params = [];

    /**
     * Get the PaymentSettings instance.
     */
    public function getPaymentSettingsProperty()
    {
        return app(PaymentSettings::class);
    }

    /**
     * Get the GeneralSettings instance.
     */
    public function getGeneralSettingsProperty()
    {
        return app(GeneralSettings::class);
    }

    /**
     * Get the PhonePeSettings instance.
     */
    public function getPhonePeSettingsProperty()
    {
        return app(PhonePeSettings::class);
    }

    /**
     * Initialize the component with the provided ID.
     *
     * @param mixed $id
     */
    public function mount($id)
    {
        $this->id = $id;
    }

    /**
     * Create a new PhonePe payment transaction.
     */
    public function createPayment()
    {
        $phonePeSettings = $this->getPhonePeSettingsProperty();
        $merchantId = $phonePeSettings->merchantId;
        $saltKey = $phonePeSettings->saltKey;
        $saltIndex = $phonePeSettings->saltIndex;
        $mode = $phonePeSettings->env_mode;

        $transactionId = $this->generateTransactionId();
        $this->handleWebhookUpgrade($transactionId);

        $phonePe = PhonePe::init(
            $merchantId,
            auth()->id(),
            $saltKey,
            $saltIndex,
            route('phone-pe.redirect', $transactionId),
            route('phone-pe.callback', $transactionId),
            $mode
        );

        $amountInPaisa = $this->convertToPaisa($this->total);
        $userMobile = $this->getUserMobile();

        $redirectURL = $phonePe->standardCheckout()->createTransaction(
            $amountInPaisa,
            $userMobile,
            $transactionId,
            route('phone-pe.redirect', $transactionId),
            route('phone-pe.callback', $transactionId)
        )->getTransactionURL();

        return redirect()->to($redirectURL);
    }

    /**
     * Generate a transaction ID based on type.
     */
    private function generateTransactionId(): string
    {
        return ($this->type === 'PKG' ? "PKG-" : 'MERCHANT-') . rand(100000, 999999);
    }

    /**
     * Convert the total amount to paisa.
     */
    private function convertToPaisa(float $amount): int
    {
        return (int)($amount * 100);
    }

    /**
     * Get the user's mobile number or a default value.
     */
    private function getUserMobile(): string
    {
        $user = auth()->user();
        return $user->phone_number ?? $user->whatsapp_number ?? '9999999999';
    }

    /**
     * Handle webhook upgrade logic.
     */
    protected function handleWebhookUpgrade(string $order_id)
    {
        $data = [
            'data' => json_encode($this->data),
            'payment_id' => $order_id,
            'payment_method' => 'phonepe',
            'status' => 'pending',
        ];

        try {
            if ($this->type === 'PKG') {
                WebhookPackage::create($data);
            } else {
                WebhookUpgrade::create($data);
            }
        } catch (\Throwable $exception) {
            logger()->error('Webhook upgrade failed', [
                'error' => $exception->getMessage(),
                'data' => $data,
            ]);
        }
    }

    /**
     * Render the component view.
     */
    public function render()
    {
        return view('phonepe::phonepe-payment');
    }
}
