<?php

namespace Adfox\ECommerce\Filament\App\Resources;

use Adfox\ECommerce\Filament\App\Resources\WarehouseResource\Pages;
use Adfox\ECommerce\Filament\App\Resources\WarehouseResource\RelationManagers\InventoryRelationManager;
use Adfox\ECommerce\Filament\App\Resources\WarehouseResource\RelationManagers\StockTransactionRelationManager;
use Adfox\ECommerce\Models\Warehouse as ModelsWarehouse;
use Filament\Resources\Resource;
use App\Models\Reservation\Location;
use Adfox\ECommerce\Models\Product;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\Action;
use Filament\Notifications\Notification;
use App\Models\City;
use App\Models\Country;
use App\Models\State;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Forms\Components\Actions\Action as ActionsAction;
use Filament\Forms\Components\Group;
use Filament\Forms\Form;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\EditAction;

class WarehouseResource extends Resource
{
    protected static ?string $model = ModelsWarehouse::class;

    public static function getNavigationGroup(): ?string
    {
        return __('messages.t_ecommerce_navigation');
    }

    public static function isDiscovered(): bool
    {
        // return is_ecommerce_active();
        return false;
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema(self::getFormColumns());
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultSort('created_at', 'desc')
            ->query(ModelsWarehouse::where('user_id', auth()->id()))
            ->emptyStateIcon('/images/not-found.svg')
            // ->emptyStateHeading(__('messages.t_no_ads'))
            ->description(false)
            ->columns(self::getTableColumns())
            ->filters([])
            ->actions(self::getTableActions());
    }

    public static function createWarehouseAction(): Action
    {
        return Action::make('Create Warehouse')
            ->form(self::getFormColumns())
            ->action(function ($data) {
                $data['user_id'] = auth()->id();
                $warehouse = ModelsWarehouse::create($data);

                if ($warehouse) {
                    Notification::make()
                        ->title('Warehouse ' . __('messages.t_saved_successfully'))
                        ->success()
                        ->send();
                }
            });
    }

    private static function getTableColumns(): array
    {
        return [
            TextColumn::make('name')->label(__('Name')),
            TextColumn::make('capacity')->label(__('Capacity')),
            TextColumn::make('location.name')->label(__('Location')),
        ];
    }

    private static function getTableActions(): array
    {
        return [
            self::manageInventoryAction(),
            EditAction::make('edit')
                ->icon('heroicon-o-pencil-square')
                ->label(__('messages.t_edit')),
            DeleteAction::make('delete')
        ];
    }

    private static function manageInventoryAction(): Action
    {
        return Action::make('Manage inventory')
            ->label(__('messages.t_manage_inventory'))
            ->color('success')
            ->form(fn($record) => self::getInventoryForm($record))
            ->action(fn($data, $record) => self::updateInventory($data, $record));
    }

    private static function getInventoryForm($record): array
    {
        return [
            Select::make('ad_id')
                ->options(Product::where('user_id', auth()->id())->pluck('title', 'id'))
                ->label('Product')
                ->live(onBlur: true)
                ->afterStateUpdated(fn(Get $get, Set $set) => self::setInventoryState($get, $set, $record))
                ->required(),
            self::getQuantityField('quantity_in_stock', 'Remove quantity', 'remove_quantity', '-'),
            TextInput::make('reorder_level')->numeric()->minValue(1)->required(),
        ];
    }

    private static function getQuantityField(string $field, string $removeLabel, string $removeField, string $operator): TextInput
    {
        return TextInput::make($field)
            ->required()
            ->readOnly()
            ->live(onBlur: true)
            ->prefixAction(
                ActionsAction::make($removeLabel)
                    ->icon("heroicon-m-minus")
                    ->form([TextInput::make($removeField)->numeric()->minValue(1)->required()])
                    ->action(fn($data, Get $get, Set $set) => self::adjustStock($data, $get, $set, $field, -1))
            )
            ->suffixAction(
                ActionsAction::make('Add quantity')
                    ->icon('heroicon-m-plus')
                    ->form([TextInput::make('new_quantity')->numeric()->minValue(1)->required()])
                    ->action(fn($data, Get $get, Set $set) => self::adjustStock($data, $get, $set, $field, 1))
            );
    }

    private static function setInventoryState(Get $get, Set $set, $record)
    {
        $inventory = $record->inventories()->where('ad_id', $get('ad_id'))->where('user_id', auth()->id())->first();
        $set('quantity_in_stock', $inventory->quantity_in_stock ?? 0);
        $set('reorder_level', $inventory->reorder_level ?? null);
    }

    private static function adjustStock($data, Get $get, Set $set, $field, $direction)
    {
        if (!isset($data['new_quantity'])) {
            $data['new_quantity'] = 1;
        }

        $currentStock = $get($field);
        $adjustment = $data['new_quantity'] ?? $data['remove_quantity'];
        $newStock = max(0, $currentStock + ($adjustment * $direction));
        $set($field, $newStock);
    }

    private static function updateInventory($data, $record)
    {
        $currentQuantity = $record->inventories()->sum('quantity_in_stock');
        $remainingQuantity = abs($record->capacity - $currentQuantity);

        if ($data['quantity_in_stock'] > $remainingQuantity) {
            Notification::make()
                ->title('You are nearing your warehouse capacity. Your current capacity level is ' . $record->capacity . ', with ' . $remainingQuantity . ' units of space remaining')
                ->info()
                ->send();

            return;
        }

        $data['user_id'] = auth()->id();

        $inventory = $record->inventories()->where('ad_id', $data['ad_id'])->where('user_id', auth()->id())->first();
        $stockTransactions = [
            'ad_id' => $data['ad_id'],
            'user_id' => $data['user_id'],
            'quantity' => $data['quantity_in_stock'],
            'transaction_date' => now(),
            'transaction_type' => $inventory ? ($data['quantity_in_stock'] > $inventory->quantity_in_stock ? 'purchase' : 'adjustment') : 'purchase',
        ];

        if ($inventory) {
            $inventory->update($data);
        } else {
            $record->inventories()->create($data);
        }

        $record->stock_transactions()->create($stockTransactions);

        Notification::make()
            ->title(__('messages.t_saved_successfully'))
            ->success()
            ->send();
    }

    private static function updateWarehouse($data, $record)
    {
        $record->update($data);

        Notification::make()
            ->title(__('messages.t_updated_successfully'))
            ->success()
            ->send();
    }

    public static function getFormColumns(): array
    {
        return [
            TextInput::make('name')->minLength(5)->maxLength(100)->required(),
            TextInput::make('capacity')->numeric()->minValue(1)->required(),
            self::locationSelectField(),
        ];
    }

    private static function locationSelectField(): Select
    {
        return Select::make('location_id')
            ->label('Location')
            ->options(fn() => Location::where('user_id', auth()->id())->where('type', 'warehouse')->pluck('name', 'id'))
            ->disableOptionWhen(fn($value) => ModelsWarehouse::where('user_id', auth()->id())->pluck('location_id')->contains($value))
            ->required()
            ->suffixAction(self::createLocationAction());
    }

    private static function createLocationAction(): ActionsAction
    {
        return ActionsAction::make('Create Location')
            ->icon('heroicon-m-plus')
            ->form(self::getLocationForm())
            ->action(function ($data, Set $set) {
                $data['user_id'] = auth()->id();
                $location = Location::create($data);

                Notification::make()
                    ->title('Location ' . __('messages.t_saved_successfully'))
                    ->success()
                    ->send();
                $set('location_id', $location->id);
            });
    }

    public static function getLocationForm(): array
    {
        return [
            Group::make()
                ->schema([
                    TextInput::make('name')->required(),
                    TextInput::make('address')->required(),
                    self::countrySelectField(),
                    self::stateSelectField(),
                    self::citySelectField(),
                    TextInput::make('postal_code')->required(),
                ])->columns(2)
        ];
    }

    private static function countrySelectField(): Select
    {
        return Select::make('country_id')
            ->label('Country')
            ->options(Country::orderBy('name')->pluck('name', 'id')->toArray())
            ->live()
            ->afterStateUpdated(fn(callable $set) => $set('state_id', null))
            ->required();
    }

    private static function stateSelectField(): Select
    {
        return Select::make('state_id')
            ->label('State')
            ->options(fn(Get $get) => $get('country_id') ? State::where('country_id', $get('country_id'))->orderBy('name')->pluck('name', 'id')->toArray() : [])
            ->live()
            ->hidden(fn(Get $get) => !$get('country_id'))
            ->afterStateUpdated(fn(callable $set) => $set('city_id', null))
            ->required();
    }

    private static function citySelectField(): Select
    {
        return Select::make('city_id')
            ->label('City')
            ->options(fn(Get $get) => $get('state_id') ? City::where('state_id', $get('state_id'))->orderBy('name')->pluck('name', 'id')->toArray() : [])
            ->hidden(fn(Get $get) => !$get('state_id'))
            ->required();
    }

    public static function getRelations(): array
    {
        return [
            InventoryRelationManager::class,
            StockTransactionRelationManager::class
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListWarehouses::route('/'),
            // 'create' => Pages\CreateWarehouse::route('/create'),
            'edit' => Pages\EditWarehouse::route('/{record}/edit'),
        ];
    }
}
