<?php

namespace Adfox\LiveChat\Filament\Pages;

use App\Filament\Clusters\CommunicationSettings;
use App\Settings\LiveChatSettings;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Pages\SettingsPage;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Config;
use BezhanSalleh\FilamentShield\Traits\HasPageShield;

class LiveChat extends SettingsPage
{
    use HasPageShield;

    protected static ?string $cluster = CommunicationSettings::class;
    protected static ?string $navigationIcon = 'heroicon-o-chat-bubble-bottom-center';
    protected static ?string $slug = 'manage-livechat-settings';
    protected static string $settings = LiveChatSettings::class;
    protected static ?int $navigationSort = 3;

    public static function getNavigationBadgeTooltip(): ?string
    {
        return 'Live Chat Module (Addon) - Enable real-time messaging between buyers and sellers for instant communication. Requires separate purchase.';
    }

    public static function getNavigationBadge(): ?string
    {
        $isDemo = Config::get('app.demo');
        return  $isDemo  ? 'Addon' : '';
    }

    public static function getNavigationBadgeColor(): ?string
    {
        return 'danger';
    }

    public static function getNavigationLabel(): string
    {
        return __('messages.t_ap_live_chat');
    }

    public function getTitle(): string
    {
        return __('messages.t_ap_live_chat');
    }

    public static function canAccess(): bool
    {
        return userHasPermission('page_LiveChat');
    }

    protected function mutateFormDataBeforeSave(array $data): array
    {
        $previousData = app(LiveChatSettings::class);
        $filtered = [];

        foreach ($data as $key => $value) {
            if (property_exists($previousData, $key)) {
                $this->updateConfig($key, $value);
                $filtered[$key] = $value ?? ($this->isStringProperty($previousData->$key) ? '' : $value);
            } else {
                $filtered[$key] = $value;
            }
        }

        Artisan::call('config:clear');
        return $filtered;
    }

    private function updateConfig(string $key, $value): void
    {
        $configMapping = [
            'pusher_key' => 'chatify.pusher.key',
            'pusher_secret' => 'chatify.pusher.secret',
            'pusher_app_id' => 'chatify.pusher.app_id',
            'pusher_cluster' => 'chatify.pusher.options.cluster',
            'encryption' => 'chatify.pusher.options.encrypted',
        ];

        if (isset($configMapping[$key])) {
            Config::write($configMapping[$key], $value);
        }
    }

    private function isStringProperty($property): bool
    {
        return gettype($property) === 'string';
    }

    public function form(Form $form): Form
    {
        $isDemo = Config::get('app.demo');

        return $form->schema([
            Toggle::make('enable_livechat')
                ->label(__('messages.t_ap_enable_livechat'))
                ->helperText(__('messages.t_ap_enable_livechat_helper')),

            $this->conditionalInput('pusher_key', $isDemo),
            $this->conditionalInput('pusher_secret', $isDemo),
            $this->conditionalInput('pusher_app_id', $isDemo),

            TextInput::make('pusher_cluster')
                ->label(__('messages.t_ap_pusher_cluster'))
                ->required()
                ->helperText(__('messages.t_ap_pusher_cluster_helper')),

            Toggle::make('encryption')
                ->label(__('messages.t_ap_enable_encryption'))
                ->helperText(__('messages.t_ap_enable_encryption_helper')),

            Toggle::make('enable_uploading_attachments')
                ->label(__('messages.t_ap_enable_uploading_attachments'))
                ->helperText(__('messages.t_ap_enable_uploading_attachments_helper')),

            Toggle::make('enable_emojis')
                ->label(__('messages.t_ap_enable_emojis'))
                ->helperText(__('messages.t_ap_enable_emojis_helper')),

            Toggle::make('enable_audio_recording')
                ->label(__('messages.t_ap_enable_audio_recording'))
                ->helperText(__('messages.t_ap_enable_audio_recording_helper')),
            //Todo: Enable when sound feature is implemented
            // Toggle::make('play_new_message_sound')
            //     ->label(__('messages.t_ap_play_new_message_sound'))
            //     ->helperText(__('messages.t_ap_play_new_message_sound_helper')),

            TextInput::make('allowed_image_extensions')
                ->label(__('messages.t_ap_allowed_image_extensions'))
                ->required()
                ->helperText(__('messages.t_ap_allowed_image_extensions_helper')),

            TextInput::make('allowed_file_extensions')
                ->label(__('messages.t_ap_allowed_file_extensions'))
                ->required()
                ->helperText(__('messages.t_ap_allowed_file_extensions_helper')),

            TextInput::make('max_file_size')
                ->label(__('messages.t_ap_max_file_size'))
                ->numeric()
                ->required()
                ->helperText(__('messages.t_ap_max_file_size_helper')),
        ])->columns(2);
    }

    private function conditionalInput(string $key, bool $isDemo): Placeholder|TextInput
    {
        return $isDemo
            ? Placeholder::make($key)
                ->label(__('messages.t_ap_' . $key))
                ->content('*****')
                ->hint(__('messages.t_ap_' . $key . '_hint'))
            : TextInput::make($key)
                ->label(__('messages.t_ap_' . $key))
                ->required()
                ->helperText(__('messages.t_ap_' . $key . '_helper'));
    }
}
