<?php

namespace Adfox\Subscription\Http\Controllers;

use Adfox\Subscription\Jobs\HandleInvoiceCreatedJob;
use Adfox\Subscription\Jobs\HandleInvoicePaymentFailedJob;
use Adfox\Subscription\Jobs\HandleInvoicePaymentSucceededJob;
use Adfox\Subscription\Jobs\HandleSubscriptionCreatedJob;
use Adfox\Subscription\Jobs\HandleSubscriptionDeletedJob;
use Adfox\Subscription\Jobs\HandleSubscriptionUpdatedJob;
use Adfox\Subscription\Services\InvoiceService;
use Adfox\Subscription\Services\SubscriptionService;
use App\Settings\SubscriptionSettings;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Stripe\Webhook;
use Symfony\Component\HttpFoundation\Response;

class StripeWebhookController extends Controller
{

    public $subscriptionService;
    public $invoiceService;

    public function __construct(SubscriptionService $subscriptionService, InvoiceService $invoiceService)
    {
        $this->subscriptionService = $subscriptionService;
        $this->invoiceService = $invoiceService;
    }

    public function handleWebhook(Request $request)
    {
        // Set Stripe webhook secret key
        $stripeWebhookSecret = app(SubscriptionSettings::class)->stripe_webhook_secret;
        $stripeSignature = $request->header('Stripe-Signature');
        $payload = $request->getContent();
        $event = null;

        try {
            $event = Webhook::constructEvent($payload, $stripeSignature, $stripeWebhookSecret);
        } catch (\UnexpectedValueException $e) {
            // Invalid payload
            return response()->json(['error' => 'Invalid payload'], Response::HTTP_BAD_REQUEST);
        } catch (\Stripe\Exception\SignatureVerificationException $e) {
            // Invalid signature
            return response()->json(['error' => 'Invalid signature'], Response::HTTP_BAD_REQUEST);
        }

        $responseData = $event->data->object;

        // Handle the event
        switch ($event->type) {
            case 'customer.subscription.created':
                HandleSubscriptionCreatedJob::dispatch($responseData)->onQueue('high');
                break;
            case 'customer.subscription.updated':
                HandleSubscriptionUpdatedJob::dispatch($responseData)->onQueue('default');
                break;
            case 'customer.subscription.deleted':
                HandleSubscriptionDeletedJob::dispatch($responseData)->onQueue('default');
                break;
            case 'invoice.created':
                HandleInvoiceCreatedJob::dispatch($responseData)->onQueue('default');
                break;

            case 'invoice.payment_succeeded':
                HandleInvoicePaymentSucceededJob::dispatch($responseData)->onQueue('default');
                break;

            case 'invoice.payment_failed':
                HandleInvoicePaymentFailedJob::dispatch($responseData)->onQueue('default');
                break;
            //TODO: upcoming feature
            // case 'invoice.paid':
            // $this->handleInvoicePaid($invoice);
            //     break;
            // case 'invoice.upcoming':
            //     break;
            // case 'entitlements.active_entitlement_summary.updated':
            //     // Then define and call a method to handle active entitlement summary updated.
            //     // handleEntitlementUpdated($subscription);
            //     break;
            // case 'customer.subscription.trial_will_end':
            //     // contains a \Stripe\Subscription
            //    // handleTrialWillEnd($subscription);
            //    break;
            default:
                // Unexpected event type
                return response()->json(['error' => 'Unhandled event type'], 400);
        }
        
        return response()->json(['success' => 'Webhook handled'], 200);
    }
}
