<?php

namespace Adfox\Subscription\Livewire\Promotion;

use App\Models\Promotion;
use App\Settings\{
    FlutterwaveSettings,
    OfflinePaymentSettings,
    PaymongoSettings,
    PaymentSettings,
    PaypalSettings,
    PaystackSettings,
    PayuSettings,
    PhonePeSettings,
    RazorpaySettings,
    StripeSettings
};
use Filament\Forms\Components\Radio;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Akaunting\Money\Currency as AkauntingCurrency;
use Livewire\Component;
use Livewire\Attributes\Reactive;

class PaymentPromotion extends Component implements HasForms
{
    use InteractsWithForms;

    public $currentPayment;
    public $payment_method;
    public $promotionIds;
    public $type = 'UG';
    public $promotions;
    public $selectedPromotions = [];
    public $subtotal = 0;
    public $tax = 0;
    public $total = 0;
    public $isDifferentRate = false;
    public $convertedTotal = 0;
    public $defaultCurrency;

    #[Reactive]
    public $selectedAdSlugs = [];

    protected $paymentGateways = [
        'stripe' => 'payment.stripe-payment',
        'paypal' => 'paypal-payment',
        'flutterwave' => 'flutterwave-payment',
        'offline' => 'offline-payment',
        'paymongo' => 'paymongo-payment',
        'razorpay' => 'razorpay-payment',
        'paystack' => 'paystack-payment',
        'phonepe' => 'phonepe-payment',
        'payu' => 'payu-payment',
    ];

    public $id;

    public function mount()
    {
        $this->initializePaymentOptions();
        $this->updatePaymentData();
    }

    protected function updatePaymentData()
    {
        $this->promotions = Promotion::whereIn('id', $this->selectedPromotions)->get();
        $totalPromotionPrice = $this->promotions->sum('price');
        $adCount = count($this->selectedAdSlugs);
        $this->subtotal = $totalPromotionPrice * $adCount;

        $paymentSettings = app(PaymentSettings::class);

        $this->tax = $this->calculateTax($paymentSettings);
        $this->total = $this->subtotal + $this->tax;

        $this->defaultCurrency = $paymentSettings->currency;
        $gatewayRate = $this->getPaymentGatewayRate();
        $systemRate = $paymentSettings->exchange_rate;

        $this->isDifferentRate = $gatewayRate !== 1.0 && $gatewayRate !== $systemRate;
        $this->convertedTotal = $this->total * $gatewayRate / $systemRate;
    }

    private function calculateTax($paymentSettings)
    {
        if (!$paymentSettings->enable_tax) {
            return 0;
        }

        return $paymentSettings->tax_type === 'percentage'
            ? ($this->subtotal * $paymentSettings->tax_rate) / 100
            : $paymentSettings->tax_rate;
    }

    public function getPaymentDataProperty()
    {
        return [
            'user_id' => auth()->id(),
            'ad_id' => $this->id,
            'promotionIds' => $this->selectedPromotions,
            'subtotal' => $this->subtotal,
            'tax' => $this->tax,
            'total' => $this->total,
            'selected_ads' => $this->selectedAdSlugs,
        ];
    }

    private function getPaymentGatewayRate()
    {
        return match ($this->payment_method) {
            'stripe' => app(StripeSettings::class)->exchange_rate,
            'paypal' => app(PaypalSettings::class)->exchange_rate,
            'flutterwave' => app(FlutterwaveSettings::class)->exchange_rate,
            'offline' => app(OfflinePaymentSettings::class)->exchange_rate,
            'paymongo' => app(PaymongoSettings::class)->exchange_rate,
            'paystack' => app(PaystackSettings::class)->exchange_rate,
            'razorpay' => app(RazorpaySettings::class)->exchange_rate,
            'phonepe' => app(PhonePeSettings::class)->exchange_rate,
            'payu' => app(PayuSettings::class)->exchange_rate,
            default => 1.0,
        };
    }

    public function getExchangeCurrencySymbol()
    {
        return match ($this->payment_method) {
            'stripe' => (new AkauntingCurrency(app(StripeSettings::class)->currency))->getSymbol(),
            'paypal' => (new AkauntingCurrency(app(PaypalSettings::class)->currency))->getSymbol(),
            'flutterwave' => (new AkauntingCurrency(app(FlutterwaveSettings::class)->currency))->getSymbol(),
            'offline' => (new AkauntingCurrency(app(OfflinePaymentSettings::class)->currency))->getSymbol(),
            'paymongo' => (new AkauntingCurrency(app(PaymongoSettings::class)->currency))->getSymbol(),
            'paystack' => (new AkauntingCurrency(app(PaystackSettings::class)->currency))->getSymbol(),
            'razorpay' => (new AkauntingCurrency(app(RazorpaySettings::class)->currency))->getSymbol(),
            'phonepe' => (new AkauntingCurrency(app(PhonePeSettings::class)->currency))->getSymbol(),
            'payu' => (new AkauntingCurrency(app(PayuSettings::class)->currency))->getSymbol(),
            default => '$',
        };
    }

    protected function initializePaymentOptions()
    {
        $paymentOptions = [];

        foreach ([
            'stripe' => StripeSettings::class,
            'paypal' => PaypalSettings::class,
            'flutterwave' => FlutterwaveSettings::class,
            'offline' => OfflinePaymentSettings::class,
            'paymongo' => PaymongoSettings::class,
            'razorpay' => RazorpaySettings::class,
            'paystack' => PaystackSettings::class,
            'payu' => PayuSettings::class,
        ] as $key => $class) {
            if (app($class)->status) {
                $paymentOptions[$key] = app($class)->name;
            }
        }

        if (count($paymentOptions) === 1) {
            $this->payment_method = array_key_first($paymentOptions);
            $this->currentPayment = $this->paymentGateways[$this->payment_method];
        }

        return $paymentOptions;
    }

    public function form(Form $form): Form
    {
        $paymentOptions = $this->initializePaymentOptions();

        return $form->schema([
            Radio::make('payment_method')
                ->hiddenLabel()
                ->live()
                ->options($paymentOptions)
                ->afterStateUpdated(function ($state) {
                    $this->currentPayment = $this->paymentGateways[$state] ?? null;
                    $this->updatePaymentData();
                }),
        ]);
    }

    public function render()
    {
        return view('subscription::promotion.payment-promotion');
    }
}
