<?php

namespace Adfox\Subscription\Filament\Resources;

use Adfox\Subscription\Filament\Resources\PlanResource\Pages;
use App\Filament\Clusters\SubscriptionManagement;
use App\Models\Plan;
use App\Models\Promotion;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Resources\Concerns\Translatable;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ToggleColumn;
use Filament\Tables\Table;
use Filament\Forms\Components\Component;
use Filament\Forms\Components\Contracts\CanDisableOptions;
use Filament\Forms\Components\Section;
use Filament\Forms\Set;
use Illuminate\Support\HtmlString;
use Illuminate\Support\Str;
use BezhanSalleh\FilamentShield\Contracts\HasShieldPermissions;


class PlanResource extends Resource implements HasShieldPermissions
{
    use Translatable;

    protected static ?string $cluster = SubscriptionManagement::class;

    protected static ?string $navigationIcon = 'heroicon-o-list-bullet';

    protected static ?string $model = Plan::class;


    public static function getModelLabel(): string
    {
        return __('messages.t_ap_plan');
    }

    public static function getPermissionPrefixes(): array
    {
        return [
            'create',
            'update',
            'view_any',
            'delete_any',
        ];
    }

    public static function canViewAny(): bool
    {
        return userHasPermission('view_any_plan');
    }

    public static function canCreate(): bool
    {
        return userHasPermission('create_plan');
    }

    public static function canEdit($record): bool
    {
        return userHasPermission('update_plan');
    }

    public static function canDeleteAny(): bool
    {
        return userHasPermission('delete_any_plan');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('name')
                    ->required()
                    ->maxLength(80)
                    ->unique()
                    ->live(onBlur: true)
                    ->afterStateUpdated(fn(Set $set, ?string $state) => $set('slug', Str::slug($state)))
                    ->label(__('messages.t_ap_name'))
                    ->helperText(__('messages.t_ap_name_helper_text')),

                Hidden::make('slug'),

                Toggle::make('is_free')
                    ->live()
                    ->afterStateUpdated(function (Set $set, ?bool $state) {
                        if(!$state){
                            $set('delete_free_ads', false);
                            $set('free_listing_duration_days', null);
                        }
                    })
                    ->label(__('messages.t_ap_free')),

                Toggle::make('is_active')
                    ->label(__('messages.t_ap_active'))
                    ->inline(false),

                TextInput::make('price')
                    ->required()
                    ->helperText(__('messages.t_ap_price_helper_text'))
                    ->numeric()
                    ->minValue(0)
                    ->label(__('messages.t_ap_price')),

                Textarea::make('description')
                    ->required()
                    ->minLength(10)
                    ->maxLength(120)
                    ->columnSpanFull()
                    ->label(__('messages.t_ap_description'))
                    ->helperText(__('messages.t_ap_description_helper_text')),

                Section::make(__('messages.t_ap_ad_count'))
                    ->columns(2)
                    ->description(__('messages.t_ap_ad_count_helper_text'))
                    ->schema([
                        TextInput::make('ad_count')
                            ->numeric()
                            ->minValue(0)
                            ->label(__('messages.t_ap_ad_count'))
                            ->default(0)
                            ->helperText(__('messages.t_ap_ad_count_helper_text')),

                        TextInput::make('feature_ad_count')
                            ->numeric()
                            ->minValue(0)
                            ->label(__('messages.t_ap_feature_ad_count'))
                            ->default(0)
                            ->helperText(__('messages.t_ap_feature_ad_count_helper_text')),

                        TextInput::make('urgent_ad_count')
                            ->numeric()
                            ->minValue(0)
                            ->label(__('messages.t_ap_urgent_ad_count'))
                            ->default(0)
                            ->helperText(__('messages.t_ap_urgent_ad_count_helper_text')),

                        TextInput::make('spotlight_ad_count')
                            ->numeric()
                            ->minValue(0)
                            ->label(__('messages.t_ap_spotlight_ad_count'))
                            ->default(0)
                            ->helperText(__('messages.t_ap_spotlight_ad_count_helper_text')),

                        TextInput::make('website_url_count')
                            ->numeric()
                            ->minValue(0)
                            ->label(__('messages.t_ap_website_url_count'))
                            ->default(0)
                            ->helperText(__('messages.t_ap_website_url_count_helper_text')),
                    ]),

                Section::make(__('messages.t_ap_listings'))
                    ->collapsible()
                    ->columns(2)
                    ->description(__('messages.t_ap_listings_helper_text'))
                    ->schema([
                        TextInput::make('images_limit')
                            ->label(__('messages.t_ap_images_limit'))
                            ->numeric()
                            ->helperText(__('messages.t_ap_images_limit_helper_text'))
                            ->default(2),

                        Toggle::make('video_posting')
                            ->label(__('messages.t_ap_video_posting'))
                            ->helperText(__('messages.t_ap_video_posting_helper_text'))
                            ->default(false),

                        Select::make('ads_level')
                            ->label(__('messages.t_ap_ads_level'))
                            ->helperText(__('messages.t_ap_ads_level_helper_text'))
                            ->options([
                                'not_available' => __('messages.t_ap_not_available'),
                                'basic' => __('messages.t_ap_basic'),
                                'advanced' => __('messages.t_ap_advanced'),
                            ])
                            ->default('not_available'),

                        Toggle::make('delete_free_ads')
                            ->visible(fn(Get $get)=>$get('is_free') )
                            ->live()
                            ->label(__('messages.t_ap_delete_free_ads'))
                            ->helperText(__('messages.t_ap_delete_free_ads_helper'))
                            ->default(false),

                        TextInput::make('free_listing_duration_days')
                            ->visible(fn(Get $get)=>$get('is_free') && $get('delete_free_ads'))
                            ->label(__('messages.t_ap_free_listing_duration_days'))
                            ->helperText(__('messages.t_ap_free_listing_duration_days_helper'))
                            ->numeric()
                            ->minValue(1)
                            ->nullable(),
                    ]),


                // User/Store Page Customization Section
                Section::make(__('messages.t_ap_user_store_page_customization'))
                    ->collapsible()
                    ->columns(2)
                    ->description(__('messages.t_ap_manage_customization_options'))
                    ->schema([
                        TextInput::make('banner_count')
                            ->label(__('messages.t_ap_banner_count'))
                            ->numeric()
                            ->helperText(__('messages.t_ap_enable_banners_with_or_without_links'))
                            ->default(0),

                        Toggle::make('enable_user_profile_description')
                            ->label(__('messages.t_ap_profile_description'))
                            ->helperText(__('messages.t_ap_allow_profile_customization_with_description_and_social_media_links'))
                            ->default(true),

                        Toggle::make('enable_social_media_links')
                            ->label(__('messages.t_ap_social_media_links'))
                            ->helperText(__('messages.t_ap_enable_disable_social_media_links_for_user_profiles'))
                            ->default(false),

                        Toggle::make('enable_location')
                            ->label(__('messages.t_ap_enable_location'))
                            ->helperText(__('messages.t_ap_allow_displaying_location_details_on_store_or_user_profile'))
                            ->default(false),

                        Toggle::make('rating')
                            ->label(__('messages.t_ap_user_rating_5_stars'))
                            ->helperText(__('messages.t_ap_enable_star_rating_system_for_seller_profiles'))
                            ->default(true),
                            Toggle::make('enable_business_hours')
                            ->label(__('messages.t_ap_enable_business_hours'))
                            ->helperText(__('messages.t_ap_enable_business_hours_description'))
                            ->default(true),

                        Select::make('filter_options_level')
                            ->label(__('messages.t_ap_filter_options_level'))
                            ->helperText(__('messages.t_ap_provide_advanced_filters_for_store_navigation'))
                            ->options([
                                'not_available' => __('messages.t_ap_not_available'),
                                'basic' => __('messages.t_ap_basic'),
                                'advanced' => __('messages.t_ap_advanced'),
                            ])
                            ->default('not_available'),
                    ]),

                // Advanced Marketing Tools Section
                Section::make(__('messages.t_ap_advanced_marketing_tools'))
                    ->collapsible()
                    ->columns(2)
                    ->description(__('messages.t_ap_enable_advanced_marketing_options'))
                    ->schema([
                        Select::make('seo_tools_level')
                            ->label(__('messages.t_ap_seo_tools_level'))
                            ->helperText(__('messages.t_ap_allow_configuration_of_seo_elements'))
                            ->options([
                                'not_available' => __('messages.t_ap_not_available'),
                                'basic' => __('messages.t_ap_basic'),
                                'advanced' => __('messages.t_ap_advanced'),
                            ])
                            ->default('not_available'),

                        Select::make('utm_parameters_level')
                            ->label(__('messages.t_ap_utm_parameters_level'))
                            ->helperText(__('messages.t_ap_enable_utm_parameters_for_marketing_campaigns'))
                            ->options([
                                'not_available' => __('messages.t_ap_not_available'),
                                'basic' => __('messages.t_ap_basic'),
                                'advanced' => __('messages.t_ap_advanced'),
                            ])
                            ->default('not_available'),
                    ]),


                // Detailed Statistics Section
                Section::make(__('messages.t_ap_detailed_statistics'))
                    ->collapsible()
                    ->columns(2)
                    ->description(__('messages.t_ap_configure_detailed_analytics_and_insights'))
                    ->schema([
                        Toggle::make('number_of_visits')
                            ->label(__('messages.t_ap_number_of_visits'))
                            ->helperText(__('messages.t_ap_display_number_of_visits_received_by_listings'))
                            ->default(true),

                        Toggle::make('traffic_source')
                            ->label(__('messages.t_ap_traffic_source'))
                            ->helperText(__('messages.t_ap_identify_where_visitors_came_from'))
                            ->default(false),

                        Toggle::make('average_view_time')
                            ->label(__('messages.t_ap_average_view_time'))
                            ->helperText(__('messages.t_ap_show_average_time_visitors_spend_on_listings'))
                            ->default(false),

                        Toggle::make('number_of_favorites')
                            ->label(__('messages.t_ap_number_of_favorites'))
                            ->helperText(__('messages.t_ap_display_count_of_favorites'))
                            ->default(true),

                        Select::make('contact_conversion_rate_level')
                            ->label(__('messages.t_ap_contact_conversion_rate_level'))
                            ->helperText(__('messages.t_ap_calculate_percentage_of_visitors_contacted'))
                            ->options([
                                'not_available' => __('messages.t_ap_not_available'),
                                'basic' => __('messages.t_ap_basic'),
                                // 'advanced' => __('messages.t_ap_advanced'),
                            ])
                            ->default('not_available'),

                        Toggle::make('clicks_on_link')
                            ->label(__('messages.t_ap_clicks_on_link_url'))
                            ->helperText(__('messages.t_ap_show_number_of_clicks_on_external_links'))
                            ->default(false),

                        Select::make('demographic_analysis_level')
                            ->label(__('messages.t_ap_demographic_analysis_level'))
                            ->helperText(__('messages.t_ap_provide_data_on_geographic_location_age_gender'))
                            ->options([
                                'not_available' => __('messages.t_ap_not_available'),
                                'basic' => __('messages.t_ap_basic'),
                                'advanced' => __('messages.t_ap_advanced'),
                            ])
                            ->default('not_available'),

                        Select::make('product_engagement_level')
                            ->label(__('messages.t_ap_product_engagement_level'))
                            ->helperText(__('messages.t_ap_detail_interactions_banner_clicks_filter_use'))
                            ->options([
                                'not_available' => __('messages.t_ap_not_available'),
                                'basic' => __('messages.t_ap_basic'),
                                'advanced' => __('messages.t_ap_advanced'),
                            ])
                            ->default('not_available'),

                        Toggle::make('product_performance_analysis')
                            ->label(__('messages.t_ap_product_performance_analysis'))
                            ->helperText(__('messages.t_ap_compare_and_identify_most_popular_products'))
                            ->default(false),
                    ]),

                // Performance Reports Section
                Section::make(__('messages.t_ap_performance_reports'))
                    ->collapsible()
                    ->columns(2)
                    ->description(__('messages.t_ap_configure_reporting_and_automated_insights'))
                    ->schema([
                        Toggle::make('boost_analysis')
                            ->label(__('messages.t_ap_boost_analysis'))
                            ->helperText(__('messages.t_ap_analyze_effectiveness_of_boosts'))
                            ->default(false),

                        // Select::make('custom_reports_level')
                        //     ->label(__('messages.t_ap_custom_reports_level'))
                        //     ->helperText(__('messages.t_ap_allow_sellers_create_custom_reports'))
                        //     ->options([
                        //         'not_available' => __('messages.t_ap_not_available'),
                        //         'basic' => __('messages.t_ap_basic'),
                        //         'advanced' => __('messages.t_ap_advanced'),
                        //     ])
                        //     ->default('not_available'),

                        Toggle::make('automated_alerts')
                            ->label(__('messages.t_ap_automated_alerts'))
                            ->helperText(__('messages.t_ap_send_alerts_about_trends'))
                            ->default(false),
                    ]),


                // Chat Section
                Section::make(__('messages.t_ap_chat'))
                    ->collapsible()
                    ->columns(2)
                    ->description(__('messages.t_ap_configure_chat_interaction_limits_and_automation'))
                    ->schema([
                        TextInput::make('chat_limit')
                            ->label(__('messages.t_ap_monthly_chat_interaction_limit'))
                            ->numeric()
                            ->helperText(__('messages.t_ap_set_limit_on_monthly_chat_interactions'))
                            ->default(20),
                    ]),

                // Sales Automation Tools Section
                Section::make(__('messages.t_ap_sales_automation_tools'))
                    ->collapsible()
                    ->columns(2)
                    ->description(__('messages.t_ap_configure_automation_features_to_enhance_user_interactions'))
                    ->schema([
                        Toggle::make('automated_messages')
                            ->label(__('messages.t_ap_automated_messages'))
                            ->helperText(__('messages.t_ap_configure_automatic_responses_for_common_interactions'))
                            ->default(false),

                        Toggle::make('automated_email_marketing')
                            ->label(__('messages.t_ap_automated_email_marketing'))
                            ->helperText(__('messages.t_ap_send_automated_email_campaigns_based_on_user_behavior'))
                            ->default(false),
                    ]),
                Hidden::make('invoice_interval')
                    ->default('month'),
                Hidden::make('invoice_period')
                    ->default(1)

                // Repeater::make('features')
                //     ->columnSpanFull()
                //     ->collapsed()
                //     ->relationship()
                //     ->itemLabel(fn (array $state): ?string => $state['name'] ?? null)
                //     ->schema([
                //         TextInput::make('name')
                //             ->required()
                //             ->live(onBlur: true),
                //         Select::make('type')
                //             ->required()
                //             ->live()
                //             ->options([
                //                 'ad_count' => 'Ad Count',
                //                 'promotion' => 'Promotion'
                //             ])
                //         ->disableOptionWhen(static function (Component & CanDisableOptions $component, string $value, mixed $state) {
                //             if($value=='ad_count'){
                //             $repeater = $component->getParentRepeater();

                //                 if (! $repeater) {
                //                     return false;
                //                 }
                //                return  collect($repeater->getState())
                //                 ->pluck(
                //                     (string) str($component->getStatePath())
                //                         ->after("{$repeater->getStatePath()}.")
                //                         ->after('.'),
                //                 )
                //                 ->diff(\Arr::wrap($state))
                //                 ->filter(fn (mixed $siblingItemState): bool => filled($siblingItemState))
                //                 ->contains($value)
                //                 ;
                //             }else{
                //                 return false;
                //             }

                //             }),
                //         Select::make('promotion_id')
                //         ->visible(fn(Get $get) => $get('type') == 'promotion')
                //         ->required(fn(Get $get) => $get('type') == 'promotion')
                //         ->disableOptionsWhenSelectedInSiblingRepeaterItems()
                //         ->options(Promotion::all()->pluck('name','id'))
                //         ,
                //         TextInput::make('value')
                //             ->required()
                //             ->numeric()
                //             ->minValue(0),
                //         Textarea::make('description')
                //             ->columnSpanFull(),
                //         ]),
                //         Hidden::make('invoice_period')
                //             ->default(1)
                //         ,

                //TODO :Plan feature

                // TextInput::make('active_subscribers_limit')
                // ->helperText('The maximum number of active subscribers allowed for this plan.')
                //     ->numeric(),
                // TextInput::make('signup_fee')
                // ->helperText('A one-time fee charged at the time of signup.')
                // ->default(0),

                // TextInput::make('trial_period')
                // ->helperText('The length of the trial period'),
                // Select::make('trial_interval')
                // ->helperText('The unit of time for the trial period.')
                //     ->options([
                //         'day' => 'Day',
                //         'week' => 'Week',
                //         'month' => 'Month',
                //     ]),
                //     TextInput::make('invoice_period')
                //     ->helperText('Specify how often an invoice is generated for the subscription.(The length of the billing cycle.)')
                //     ->required(),
                // Select::make('invoice_interval')
                //     ->helperText('The unit of time for the billing cycle.')
                //     ->required()
                //     ->default('month')
                //     ->options([
                //         'month' => 'Month',
                //         'annual' => 'Annual'
                //     ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultSort('created_at', 'desc')
            ->columns([
                TextColumn::make('name')
                ->label(__('messages.t_ap_name')),
                TextColumn::make('price')
                ->label(__('messages.t_ap_price')),
                ToggleColumn::make('is_active')
                ->label(__('messages.t_ap_active')),

            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPlans::route('/'),
            'create' => Pages\CreatePlan::route('/create'),
            'edit' => Pages\EditPlan::route('/{record}/edit'),
        ];
    }
}
