<?php

namespace Adfox\Subscription\Filament\Resources;

use Adfox\Subscription\Filament\Resources\SubscriptionResource\Pages;
use Adfox\Subscription\Filament\Resources\SubscriptionResource\RelationManagers\InvoicesRelationManager;
use App\Filament\Clusters\SubscriptionManagement;
use App\Models\Plan;
use App\Models\Subscription;
use App\Models\User;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Select;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Filament\Tables\Filters\Filter;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Filters\SelectFilter;
use BezhanSalleh\FilamentShield\Contracts\HasShieldPermissions;


class SubscriptionResource extends Resource implements HasShieldPermissions
{
    protected static ?string $model = Subscription::class;

    protected static ?string $cluster = SubscriptionManagement::class;

    protected static ?string $navigationIcon = 'heroicon-o-calendar';

    public static function getModelLabel(): string
    {
        return __('messages.t_ap_subscription');
    }

    public static function getPermissionPrefixes(): array
    {
        return [
            'view_any',
            'create',
            'view',
            'delete_any'
        ];
    }

    public static function canViewAny(): bool
    {
        return userHasPermission('view_any_subscription');
    }

    public static function canCreate(): bool
    {
        return userHasPermission('create_subscription');
    }

    public static function canView($record): bool
    {
        return userHasPermission('view_subscription');
    }

    public static function canDeleteAny(): bool
    {
        return userHasPermission('delete_any_subscription');
    }


    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Select::make('plan_id')
                    ->required()
                    ->label(__('messages.t_ap_plan'))
                    ->options(Plan::where('is_free', false)->active()->pluck('name', 'id')),

                Select::make('user_id')
                    ->required()
                    ->label(__('messages.t_ap_user'))
                    ->searchable()
                    ->options(User::all()->pluck('name', 'id')),

                DatePicker::make('ends_at')
                    ->minDate(now())
                    ->label(__('messages.t_ap_expired_at'))
                    ->required(),
                Hidden::make('status')
                    ->default('active'),

            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultSort('created_at', 'desc')
            ->columns([
                TextColumn::make('subscriber.name')
                    ->searchable()
                    ->label(__('messages.t_ap_subscriber_name')),
                TextColumn::make('plan.name')
                    ->searchable()
                    ->label(__('messages.t_ap_plan_name')),
                TextColumn::make('status')
                    ->formatStateUsing(fn(string $state): string => __("messages.t_{$state}"))
                    ->label(__('messages.t_ap_status')),
                TextColumn::make('starts_at')
                    ->date('d/m/Y')
                    ->label(__('messages.t_ap_starts_at')),
                TextColumn::make('ends_at')
                    ->date('d/m/Y')
                    ->label(__('messages.t_ap_ends_at')),
                TextColumn::make('cancels_at')
                    ->date('d/m/Y')
                    ->label(__('messages.t_ap_cancels_at')),


            ])
            ->filters([
                SelectFilter::make('status')
                    ->options([
                        'active' => __('messages.t_ap_active'),
                        'incomplete' => __('messages.t_ap_incomplete'),
                        'incomplete_expired' => __('messages.t_ap_incomplete_expired'),
                        'past_due' => __('messages.t_ap_past_due'),
                        'canceled' => __('messages.t_ap_canceled'),
                        'unpaid' => __('messages.t_ap_unpaid'),
                        'paused' => __('messages.t_ap_paused'),
                    ])
                    ->label(__('messages.t_ap_filter_status')),
                Filter::make('subscription_period')
                    ->label(__('messages.t_ap_filter_subscription_period'))
                    ->form([
                        DatePicker::make('starts_at')
                            ->label(__('messages.t_ap_start_date')),
                        DatePicker::make('ends_at')
                            ->label(__('messages.t_ap_end_date')),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['starts_at'],
                                fn(Builder $query, $date): Builder => $query->whereDate('starts_at', '>=', $date),
                            )
                            ->when(
                                $data['ends_at'],
                                fn(Builder $query, $date): Builder => $query->whereDate('ends_at', '<=', $date),
                            );
                    })
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            InvoicesRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListSubscriptions::route('/'),
            //TODO: Enable when create and edit page exist
            // 'create' => Pages\CreateSubscription::route('/create'),
            // 'edit' => Pages\EditSubscription::route('/{record}/edit'),
            'view' => Pages\ViewSubscription::route('/{record}'),
        ];
    }
}
