<?php

namespace Adfox\ECommerce\Livewire\Payment;

use App\Traits\HelperTraits;
use App\Models\WebhookPackage;
use App\Traits\HandlesPackageApplication;
use App\Traits\HandlesPromotionApplication;
use Livewire\Attributes\On;
use Livewire\Component;
use App\Models\WebhookUpgrade;
use App\Settings\PaymentSettings;
use App\Settings\RazorpaySettings;
use App\Settings\GeneralSettings;
use Razorpay\Api\Api;
use Razorpay\Api\Errors\SignatureVerificationError;

class Razorpay extends Component
{
    use HelperTraits;

    public $successUrl;
    public $totalAmount;

    /**
     * Mount the component and set the properties if an ad ID is provided.
     */
    public function mount()
    {
        $this->processPayment();
        $this->successUrl = url('callback/razorpay');
    }

    public function getPaymentSettingsProperty()
    {
        return app(PaymentSettings::class);
    }

    public function getGeneralSettingsProperty()
    {
        return app(GeneralSettings::class);
    }

    public function getRazorpaySettingsProperty()
    {
        return app(RazorpaySettings::class);
    }

    public function processPayment()
    {

        $request = new Api($this->getRazorpaySettingsProperty()->key_id, $this->getRazorpaySettingsProperty()->key_secret);

        $currency = $this->getRazorpaySettingsProperty()->currency;
        // Create order
        $order = $request->order->create([
            'amount' =>  (int)$this->totalAmount * 100,
            'currency' =>  $currency,
        ]);

        // Set Razorpay payment parameters
        $this->payment_gateway_params['order_id'] = $order->id;

        $this->handleWebhookUpgrade($order->id);
    }


    protected function handleWebhookUpgrade($order_id)
    {
        try {
            if ($this->type == 'PKG') {
                WebhookPackage::create([
                    'data' => json_encode($this->data),
                    'payment_id' => $order_id,
                    'payment_method' => 'razorpay',
                    'status' => 'pending'
                ]);
            } else {
                WebhookUpgrade::create([
                    'data' => json_encode($this->data),
                    'payment_id' => $order_id,
                    'payment_method' => 'razorpay',
                    'status' => 'pending'
                ]);
            }
        } catch (\Throwable $th) {
            // Handle any exceptions
        }
    }

    #[On('razorpay-handler')]
    public function verifyPayment($response)
    {
        $success = true;
        if (isset($response['razorpay_payment_id'])) {
            $api = new Api($this->getRazorpaySettingsProperty()->key_id, $this->getRazorpaySettingsProperty()->key_secret);
            try {
                //verify the payment
                $api->utility->verifyPaymentSignature($response);
            } catch (SignatureVerificationError $e) {
                $success = false;
                return [
                    'success' => false,
                    'message' => __('messages.t_toast_something_went_wrong')
                ];
            }
        }

        if ($success === true) {
            // Get order id
            $order_id = $response['razorpay_order_id'];
            // Check If Package Management Callback
            if ($this->type == 'PKG') {
                $orderData = WebhookPackage::where('payment_id', $order_id)
                    ->where('payment_method', 'razorpay')
                    ->where('status', 'pending')
                    ->firstOrFail();

                $routeParameters = $this->applyPackages($orderData);
                return redirect()->route('package-success', $routeParameters);
            } else {

                $orderData = WebhookUpgrade::where('payment_id', $order_id)
                    ->where('payment_method', 'razorpay')
                    ->where('status', 'pending')
                    ->firstOrFail();

                $routeParameters = $this->applyPromotionsWithRedirect($orderData);

                return redirect()->route('success-upgrade', $routeParameters);
            }
        }
    }

    /**
     * Render the component view.
     *
     * @return \Illuminate\View\View The view to render.
     */
    public function render()
    {
        return view('e-commerce::payment.razorpay');
    }
}
