<?php

namespace Adfox\Feedback\Livewire;

use App\Forms\Components\StarRating;
use App\Models\{Conversation, Feedback as ModelFeedback, User};
use App\Settings\{GeneralSettings, SEOSettings};
use Livewire\Component;
use Filament\Actions\Concerns\InteractsWithActions;
use Filament\Actions\Contracts\HasActions;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Actions\Action;
use Filament\Forms\Components\{Select, Textarea, ToggleButtons};
use Filament\Notifications\Notification;
use Filament\Support\Enums\MaxWidth;
use Illuminate\Support\Facades\Auth;
use Mokhosh\FilamentRating\Components\Rating;
use Artesaos\SEOTools\Traits\SEOTools as SEOToolsTrait;
use Livewire\Attributes\On;

class Feedback extends Component implements HasForms, HasActions
{
    use InteractsWithActions, InteractsWithForms, SEOToolsTrait;

    public $sellerId;
    public $feedbackDetails;
    public $positiveCount;
    public $neutralCount;
    public $negativeCount;
    public $sellerName;
    public $isMobileHidden = false;
    public $sellerRating;
    public $sortBy;
    public $search;
    public $feedbackCount;
    public ModelFeedback $feedback;

    public function mount($id)
    {
        // dd( $filters['sortBy']);
        $this->sellerId = $id;
        $this->verifySellerExists();
        $this->loadSellerName();
        $this->calculateFeedbackCounts();
        $this->setSeoData();
        $this->sortBy = $filters['sortBy'] ?? 'date_desc';
    }

    protected function loadSellerName()
    {
        $this->feedbackCount = User::find($this->sellerId)->feedbackCount() ?? 0;
        $this->sellerRating = User::find($this->sellerId)->rating ?? 0;
        $this->sellerName = optional(User::find($this->sellerId))->name ?? __('messages.t_unknown_seller');
    }

    /**
     * Verifies if the seller exists and if the user is not giving feedback to themselves.
     *
     * @return void
     */
    protected function verifySellerExists()
    {
        if (!User::where('id', $this->sellerId)->exists()) {
            abort(404, __('messages.t_seller_not_found'));
        }
    }

    public function feedbackAction(): Action
    {
        return Action::make('feedbackAction')
            ->hidden(function () {
                return $this->sellerId == Auth::id() ||
                       !$this->hasMinConvWithSeller() ||
                       $this->hasAlreadyLeftFeedback();
            })
            ->label(__('messages.t_leave_feedback'))
            ->form($this->feedbackFormSchema())
            ->modalWidth(MaxWidth::Large)
            ->modalSubmitActionLabel(__('messages.t_send_feedback'))
            ->action(function (array $data) {
                $this->handleFeedbackSubmission($data);
            });
    }

    protected function hasMinConvWithSeller(): bool
    {
        return Conversation::where('buyer_id', Auth::id())
            ->where('seller_id', $this->sellerId)
            ->whereHas('messages')
            ->exists();
    }

    protected function hasAlreadyLeftFeedback(): bool
    {
        return ModelFeedback::where('buyer_id', Auth::id())
            ->where('seller_id', $this->sellerId)
            ->exists();
    }

    protected function feedbackFormSchema(): array
    {
        return [
            ToggleButtons::make('experience')
                ->label(__('messages.t_rate_experience'))
                ->options([
                    'positive' => __('messages.t_positive'),
                    'neutral' => __('messages.t_neutral'),
                    'negative' => __('messages.t_negative'),
                ])
                ->icons([
                    'positive' => 'heroicon-o-face-smile',
                    'neutral' => 'heroicon-o-face-frown',
                    'negative' => 'heroicon-o-face-frown',
                ])
                ->colors([
                    'positive' => 'success',
                    'neutral' => 'warning',
                    'negative' => 'danger',
                ])
                ->inline()
                ->required()
                ->live(),
            StarRating::make('rating')
                ->label(__('messages.t_rating'))
                ->required()
                ->visible(function () {
                    if (getSubscriptionSetting('status')) {
                        return getUserSubscriptionPlan($this->sellerId)?->rating;
                    }
                    return false;
                })
                ->size('lg')
                ->helperText(__('messages.t_select_star_rating')),
            Select::make('interaction')
                ->label(__('messages.t_interaction_label'))
                ->options([
                    'successful_purchase' => __('messages.t_successful_purchase'),
                    'deal_failed'         => __('messages.t_deal_failed'),
                    'didnt_come_to_deal'  => __('messages.t_didnt_come_to_deal'),
                    'cant_reach_seller'   => __('messages.t_cant_reach_seller'),
                ])
                ->required()
                ->live(),

            Textarea::make('detail')
                ->label(__('messages.t_detailed_feedback'))
                ->required()
                ->maxLength(1000),
        ];
    }

    protected function handleFeedbackSubmission(array $data)
    {
        try {
            ModelFeedback::create([
                'buyer_id'    => Auth::id(),
                'seller_id'   => $this->sellerId,
                'experience'  => $data['experience'],
                'interaction' => $data['interaction'],
                'detail' => $data['detail'],
                'rating' => isset($data['rating']) ?(int) $data['rating'] : 0,
            ]);

            Notification::make()
                ->title(__('messages.t_feedback_submitted_successfully'))
                ->success()
                ->send();

            $this->dispatch('fetch-feedback');
        } catch (\Exception $e) {
            Notification::make()
                ->title(__('messages.t_failed_to_submit_feedback'))
                ->danger()
                ->send();
        }
    }

    protected function calculateFeedbackCounts()
    {
        $feedbacks = ModelFeedback::where('seller_id', $this->sellerId)->get();

        $this->positiveCount = $feedbacks->where('experience', 'positive')->count();
        $this->neutralCount  = $feedbacks->where('experience', 'neutral')->count();
        $this->negativeCount = $feedbacks->where('experience', 'negative')->count();
    }

    protected function setSeoData()
    {
        $generalSettings = app(GeneralSettings::class);
        $seoSettings     = app(SEOSettings::class);

        $title = __('messages.t_my_feedback') . ' ' . ($generalSettings->separator ?? '-') . ' ' .
                 ($generalSettings->site_name ?? config('app.name'));
        $this->seo()->setTitle($title);
        $this->seo()->setDescription($seoSettings->meta_description);
    }

    public function render()
    {
        return view('feedback::feedback');
    }
}
