<?php

namespace Adfox\Packages\Filament\Resources;

use Adfox\Packages\Filament\Resources\PackageResource\Pages;
use App\Filament\Clusters\PackageManagement;
use App\Models\Package;
use App\Models\Promotion;
use Filament\Forms;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Select;
use Filament\Forms\Get;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ToggleColumn;
use BezhanSalleh\FilamentShield\Contracts\HasShieldPermissions;


class PackageResource extends Resource implements HasShieldPermissions
{
    protected static ?string $model = Package::class;

    protected static ?int $navigationSort = 1;

    protected static ?string $cluster = PackageManagement::class;

    protected static ?string $navigationIcon = 'heroicon-o-square-3-stack-3d';

    public static function getPermissionPrefixes(): array
    {
        return [
            'create',
            'update',
            'view_any',
            'delete_any'
        ];
    }

    public static function canViewAny(): bool
    {
        return userHasPermission('view_any_package');
    }

    public static function canCreate(): bool
    {
        return userHasPermission('create_package');
    }

    public static function canEdit($record): bool
    {
        return userHasPermission('update_package');
    }

    public static function getModelLabel(): string
    {
        return __('messages.t_ap_package');
    }

    public static function canDeleteAny(): bool
    {
        return userHasPermission('delete_any_package');
    }


    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Toggle::make('is_default')
                    ->label(__('messages.t_ap_set_as_default'))
                    ->helperText(__('messages.t_ap_set_as_default_helper')),

                Toggle::make('is_enabled')
                    ->label(__('messages.t_ap_activate_package'))
                    ->helperText(__('messages.t_ap_activate_package_helper')),

                TextInput::make('name')
                    ->label(__('messages.t_ap_package_name'))
                    ->required()
                    ->placeholder(__('messages.t_ap_package_name_placeholder')),

                TextInput::make('duration')
                    ->label(__('messages.t_ap_duration'))
                    ->numeric()
                    ->suffix(__('messages.t_ap_days'))
                    ->required()
                    ->placeholder(__('messages.t_ap_duration_placeholder')),

                Textarea::make('features')
                    ->label(__('messages.t_ap_features'))
                    ->required()
                    ->columnSpanFull()
                    ->placeholder(__('messages.t_ap_features_placeholder'))
                    ->rows(3),

                Repeater::make('packageItems')
                    ->label(__('messages.t_ap_ad_count_options'))
                    ->relationship()
                    ->schema([
                        TextInput::make('quantity')
                            ->label(__('messages.t_ap_number_of_ads'))
                            ->numeric()
                            ->required()
                            ->placeholder(__('messages.t_ap_number_of_ads_placeholder')),

                        TextInput::make('price')
                            ->label(__('messages.t_ap_price_for_this_count'))
                            ->numeric()
                            ->required()
                            ->prefix(config('app.currency_symbol'))
                            ->placeholder(__('messages.t_ap_price_for_this_count_placeholder')),

                        Toggle::make('offer_enabled')
                            ->label(__('messages.t_ap_enable_offer'))
                            ->live()
                            ->helperText(__('messages.t_ap_enable_offer_helper')),

                        TextInput::make('offer_price')
                            ->label(__('messages.t_ap_offer_price'))
                            ->required()
                            ->numeric()
                            ->prefix(config('app.currency_symbol'))
                            ->placeholder(__('messages.t_ap_offer_price_placeholder'))
                            ->visible(fn($get) => $get('offer_enabled'))
                            ->helperText(__('messages.t_ap_offer_price_helper')),

                        DatePicker::make('offer_start')
                            ->label(__('messages.t_ap_offer_start_date'))
                            ->required()
                            ->minDate(now())
                            ->native(false)
                            ->visible(fn($get) => $get('offer_enabled'))
                            ->helperText(__('messages.t_ap_offer_start_date_helper')),

                        DatePicker::make('offer_end')
                            ->label(__('messages.t_ap_offer_end_date'))
                            ->required()
                            ->native(false)
                            ->minDate(now())
                            ->visible(fn($get) => $get('offer_enabled'))
                            ->helperText(__('messages.t_ap_offer_end_date_helper')),
                    ])
                    ->addActionLabel(__('messages.t_ap_add_ad_count_option'))
                    ->columnSpanFull()
                    ->columns(2),

                Repeater::make('packagePromotions')
                    ->label(__('messages.t_ap_promotional_options'))
                    ->relationship()
                    ->schema([
                        Select::make('promotion_id')
                            ->options(Promotion::all()->pluck('name', 'id'))
                            ->label(__('messages.t_ap_promotion_type'))
                            ->helperText(__('messages.t_ap_promotion_type_helper'))
                            ->required(),

                        Repeater::make('packageItems')
                            ->label(__('messages.t_ap_promotion_usage_limits'))
                            ->relationship()
                            ->schema([
                                TextInput::make('quantity')
                                    ->label(__('messages.t_ap_usage_limit'))
                                    ->helperText(__('messages.t_ap_usage_limit_helper'))
                                    ->numeric()
                                    ->required(),

                                TextInput::make('price')
                                    ->label(__('messages.t_ap_price_for_promotion'))
                                    ->numeric()
                                    ->required()
                                    ->prefix(config('app.currency_symbol'))
                                    ->placeholder(__('messages.t_ap_price_for_promotion_placeholder')),
                            ])
                            ->addActionLabel(__('messages.t_ap_add_promotion_usage_limit'))
                            ->columnSpanFull()
                            ->columns(2),
                    ])
                    ->addActionLabel(__('messages.t_ap_add_promotional_option'))
                    ->columnSpanFull()
                    ->columns(2)


            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultSort('created_at', 'desc')
            ->columns([
                TextColumn::make('name')
                    ->label('Package Name'),
                TextColumn::make('quantity')
                    ->hidden()
                    ->label('Ad Count'),
                TextColumn::make('duration')
                    ->suffix(' Days')
                    ->label('Duration'),
                TextColumn::make('price')
                    ->hidden()
                    ->label('Price')
                    ->prefix(config('app.currency_symbol')),
                ToggleColumn::make('is_enabled')
                    ->label('Package Status')
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPackages::route('/'),
            'create' => Pages\CreatePackage::route('/create'),
            'edit' => Pages\EditPackage::route('/{record}/edit'),
        ];
    }
}
