<?php

namespace Adfox\Subscription\Filament\Pages;

use App\Filament\Clusters\SubscriptionManagement;
use App\Settings\SubscriptionSettings;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Pages\SettingsPage;
use App\Settings\PackageSettings;
use Exception;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Section;
use Filament\Forms\Set;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\HtmlString;
use BezhanSalleh\FilamentShield\Traits\HasPageShield;

class SubscriptionSetup extends SettingsPage
{
    use HasPageShield;

    protected static ?string $slug = 'manage-subscription-settings';

    protected static ?string $cluster = SubscriptionManagement::class;

    protected static ?string $navigationIcon = 'heroicon-o-wrench';

    protected static string $settings = SubscriptionSettings::class;

    protected static ?int $navigationSort = 3; // Adjust the sort order as needed

    public static function getNavigationLabel(): string
    {
        return __('messages.t_ap_subscription_configuration');
    }

    public function getTitle(): string
    {
        return __('messages.t_ap_subscription_settings');
    }

    public static function canAccess(): bool
    {
        return userHasPermission('page_SubscriptionSetup');
    }

    protected function mutateFormDataBeforeSave(array $data): array
    {
        $previousData = app(SubscriptionSettings::class);
        $filtered = [];

        foreach ($data as $key => $item) {
            if (property_exists($previousData, $key)) {
                $propertyType = gettype($previousData->$key);

                if (is_null($item) && $propertyType === 'string') {
                    $filtered[$key] = '';
                    continue;
                }
            }
            $filtered[$key] = $item;
        }
        if (!$filtered['status']) {
            $filtered['combine_subscriptions_and_orders'] = false;
        }
        return $filtered;
    }

    public function afterSave()
    {
        //Disable package if subscription enabled
        $packageSettings = app(PackageSettings::class);
        if (isset($this->data['status']) && $this->data['status']) {
            $packageSettings->status = false;
            $packageSettings->save();
        }

        try {
            Artisan::call('paypal:initialize-plan');
        } catch (Exception $ex) {
        }
    }

    public function form(Form $form): Form
    {
        $currenciesConfig = config('money.currencies');
        $currencyCodes = array_keys($currenciesConfig);
        $isDemo = Config::get('app.demo');

        return $form
            ->schema([
                Toggle::make('status')
                    ->label(__('messages.t_ap_enable_subscription'))
                    ->live()
                    ->afterStateUpdated(fn($state, Set $set) => !$state ? $set('combine_subscriptions_and_orders', false) : '')
                    ->helperText(__('messages.t_ap_enable_subscription_helper')),

                Toggle::make('combine_subscriptions_and_orders')
                    ->helperText(__('messages.t_ap_combine_subscriptions_and_orders_helper')),

                Toggle::make('show_comparison_table')
                    ->helperText(__('messages.t_ap_show_comparison_table_helper')),

                TextInput::make('free_ad_limit')
                    ->numeric()
                    ->minValue(0)
                    ->label(__('messages.t_ap_free_ad_limit'))
                    ->placeholder(__('messages.t_ap_free_ad_limit_placeholder'))
                    ->required(),

                $isDemo ?
                Placeholder::make('stripe_webhook_secret')
                    ->label(__('messages.t_ap_stripe_webhook_secret'))
                    ->content('*****')
                    ->hint(__('messages.t_ap_stripe_webhook_secret_hint')) :
                TextInput::make('stripe_webhook_secret')
                    ->label(__('messages.t_ap_stripe_webhook_secret'))
                    ->required()
                    ->helperText(__('messages.t_ap_stripe_webhook_secret_helper'))
                    ->hint(str('['.__('messages.t_ap_stripe_webhook_hint').'](https://docs.stripe.com/webhooks)')->inlineMarkdown()->toHtmlString())
                    ->hintColor('warning')
                    ->hintIcon('heroicon-m-question-mark-circle', tooltip: __('messages.t_ap_stripe_webhook_hint_tooltip')),

                TextInput::make('ipinfo_key')
                    ->label(__('messages.t_ap_ipinfo_api_key'))
                    ->default('a15d5719db1bf2')
                    ->helperText(__('messages.t_ap_ipinfo_api_key_helper'))
                    ->required()
                    ->placeholder(__('messages.t_ap_ipinfo_api_key_placeholder')),

                Toggle::make('enable_paypal')
                    ->label(__('messages.t_ap_enable_paypal_subscription'))
                    ->live()
                    ->visible(isPaypalEnabled())
                    ->helperText(__('messages.t_ap_enable_paypal_subscription_helper')),

                Section::make(__('messages.t_ap_instructions_to_add_stripe_webhook'))
                    ->collapsed()
                    ->schema([
                        Placeholder::make('')
                            ->columnSpanFull()
                            ->content(self::getWebhookInstruction())
                    ]),

                Section::make(__('messages.t_ap_subscription_pricing'))
                    ->columns(2)
                    ->description(__('messages.t_ap_subscription_pricing_description'))
                    ->schema([
                        TextInput::make('ad_price')
                            ->label(__('messages.t_ap_price_per_ad_count'))
                            ->numeric()
                            ->helperText(__('messages.t_ap_price_per_ad_count_helper'))
                            ->required(),

                        TextInput::make('featured_ad_price')
                            ->label(__('messages.t_ap_price_per_featured_ad'))
                            ->numeric()
                            ->helperText(__('messages.t_ap_price_per_featured_ad_helper'))
                            ->required(),

                        TextInput::make('urgent_ad_price')
                            ->label(__('messages.t_ap_price_per_urgent_ad'))
                            ->numeric()
                            ->helperText(__('messages.t_ap_price_per_urgent_ad_helper'))
                            ->required(),

                        TextInput::make('spotlight_ad_price')
                            ->label(__('messages.t_ap_price_per_spotlight_ad'))
                            ->numeric()
                            ->helperText(__('messages.t_ap_price_per_spotlight_ad_helper'))
                            ->required(),

                        TextInput::make('website_url_price')
                            ->label(__('messages.t_ap_price_per_website_url'))
                            ->numeric()
                            ->helperText(__('messages.t_ap_price_per_website_url_helper'))
                            ->required(),
                    ])


            ])
            ->columns(2);
    }

    protected static function getWebhookInstruction()
    {
        return new HtmlString('<div class="prose">
        <h3>' . __('messages.t_ap_webhook_instruction_heading') . '</h3>
        <ol>
            <li>' . __('messages.t_ap_login_to_stripe_dashboard') . ':
                <ul>
                    <li>' . __('messages.t_ap_go_to_stripe_dashboard') . ' <a style="text-decoration:underline" target="blank" href="https://dashboard.stripe.com">Stripe Dashboard</a>.</li>
                    <li>' . __('messages.t_ap_login_with_stripe_account') . '</li>
                </ul>
            </li>
            <li>' . __('messages.t_ap_navigate_to_webhooks') . ':
                <ul>
                    <li>' . __('messages.t_ap_click_on_developers') . '</li>
                    <li>' . __('messages.t_ap_click_on_webhooks') . '</li>
                </ul>
            </li>
            <li>' . __('messages.t_ap_add_endpoint') . ':
                <ul>
                    <li>' . __('messages.t_ap_click_add_endpoint_button') . '</li>
                </ul>
            </li>
            <li>' . __('messages.t_ap_set_endpoint_url') . ':
                <ul>
                    <li>' . __('messages.t_ap_enter_site_url') . ' <code>' . url('/') . '/stripe/webhook</code>.</li>
                </ul>
            </li>
            <li>' . __('messages.t_ap_select_events_to_listen') . ':
                <ul>
                    <li>' . __('messages.t_ap_click_select_events') . '</li>
                    <li>' . __('messages.t_ap_choose_events') . ':
                        <ul>
                            <li><code>customer.subscription.created</code></li>
                            <li><code>customer.subscription.updated</code></li>
                            <li><code>customer.subscription.deleted</code></li>
                            <li><code>invoice.created</code></li>
                            <li><code>invoice.payment_succeeded</code></li>
                            <li><code>invoice.paid</code></li>
                            <li><code>invoice.payment_failed</code></li>
                        </ul>
                    </li>
                </ul>
            </li>
            <li>' . __('messages.t_ap_save_endpoint') . ':
                <ul>
                    <li>' . __('messages.t_ap_click_add_endpoint_button_to_save') . '</li>
                </ul>
            </li>
        </ol>
        </div>');
    }

}
