<?php
namespace Adfox\Subscription\Services;

use App\Http\Controllers\Logger;
use App\Models\Subscription;
use App\Models\User;
use Carbon\Carbon;
use App\Models\Invoice;
use Adfox\Subscription\Notifications\PaymentFailedNotification;
use Adfox\Subscription\Notifications\PaymentSucceededNotification;
use App\Models\AdPromotion;

class InvoiceService
{

    /**
     * Logs a message to a custom logger.
     *
     * @param string $message The message to log.
     * @param array  $context Additional log context.
     */
    private function logMessage($type, $text = '', $timestamp = true)
    {
        $logger = new Logger(storage_path() . '/logs/invoice.log');
        $logger->log($type, $text, $timestamp);
    }

    public function handleInvoiceCreated($invoice)
    {
        $subscription = $invoice->subscription;

        if ($subscription) {
            $subscriptionMetadata = $invoice->subscription_details->metadata;

            $user = User::find($subscriptionMetadata->user_id);
            $dueDate = $invoice->due_date ? Carbon::createFromTimestamp($invoice->due_date) : null;
            $invoiceDate = $invoice->created ? Carbon::createFromTimestamp($invoice->created) : null;
            $subscription = Subscription::where('subscription_reference', $subscription)->first();

            if ($user && $subscription) {
                Invoice::create([
                    'subscription_id' => $subscription->id,
                    'invoice_id' => $invoice->id,
                    'currency' => $invoice->currency,
                    'amount_due' => number_format($invoice->amount_due / 100, 2),
                    'amount_paid' => number_format($invoice->amount_paid / 100, 2),
                    'status' => $invoice->status,
                    'amount_remaining' => number_format($invoice->amount_remaining / 100, 2),
                    'invoice_date' => $invoiceDate,
                    'due_date' => $dueDate,
                ]);

                $this->logMessage('Invoice created for user: ' . $user->id);
            } else {
                $this->logMessage('User not found' . $user->id);
            }
        }
    }

    public function handleInvoicePaymentSucceeded($stripeInvoice)
    {
        $stripeSubscription = $stripeInvoice->subscription;

        if ($stripeSubscription && $stripeInvoice->status == 'paid' && $stripeInvoice->paid) {
            //Update: Subscription record details
            $endDate = $stripeInvoice->lines->data[0]->period->end;

            $subscription = Subscription::where('subscription_reference', $stripeSubscription)->first();

            if ($subscription) {
                $subscription->update([
                    'ends_at' => $endDate
                ]);

                $this->updateAdPromotionExpiration($subscription->id, $endDate);
            }

            //Update: Invoice record details
            $invoice = Invoice::where('invoice_id', $stripeInvoice->id)->first();

            if ($invoice) {
                $dueDate = $stripeInvoice->due_date ? Carbon::createFromTimestamp($stripeInvoice->due_date) : null;
                $invoice->update([
                    'amount_paid' => number_format($stripeInvoice->amount_paid / 100, 2),
                    'amount_remaining' => number_format($stripeInvoice->amount_remaining / 100, 2),
                    'status' => $stripeInvoice->status,
                    'due_date' => $dueDate,
                ]);

                $subscriptionMetadata = $stripeInvoice->subscription_details->metadata;
                $user = User::find($subscriptionMetadata->user_id);

                if ($user) {
                    $user->notify(new PaymentSucceededNotification($stripeInvoice));
                    $this->logMessage('Invoice payment succeed for user: ' . $user->id);
                }
            } else {
                $this->logMessage('Invoice record not found in the database');
            }
        }
    }

    public function handleInvoicePaymentFailed($invoice)
    {
        $subscription = $invoice->subscription;
        if ($subscription) {
            $subscriptionMetadata = $invoice->subscription_details->metadata;
            $user = User::find($subscriptionMetadata->user_id);
            if ($user) {
                $user->notify(new PaymentFailedNotification($invoice));
            }
        }
    }

    public function updateAdPromotionExpiration($subscriptionId, $newEndDate)
    {
        AdPromotion::where('subscription_id', $subscriptionId)
            ->update(['end_date' => $newEndDate]);
    }

}
