<?php

namespace Adfox\WalletSystem\Filament\Resources\PendingRefundResource\Pages;

use Adfox\WalletSystem\Filament\Resources\PendingRefundResource;
use Filament\Resources\Pages\ListRecords;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Filament\Resources\Components\Tab;
use App\Models\Wallets\Wallet;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Tables;

class ListPendingRefunds extends ListRecords
{
    protected static string $resource = PendingRefundResource::class;

    protected function getHeaderActions(): array
    {
        return [

        ];
    }
    public function getTabs(): array
    {
        return [
            __('messages.t_ap_buyer') => Tab::make()
                ->modifyQueryUsing(fn(Builder $query) => $query->where('type', 'buyer')->where('status', 'pending')),
            __('messages.t_ap_seller') => Tab::make()
                ->modifyQueryUsing(fn(Builder $query) => $query->where('type', 'seller')->where('status', 'pending')),
        ];
    }
    public function table(Table $table): Table
    {
        return $table

            ->modifyQueryUsing(fn(Builder $query) => $query->where('status', 'pending')->latest())
            ->columns([
                TextColumn::make('order.order_number')
                    ->label(__('messages.t_ap_order_id'))
                    ->sortable()
                    ->searchable(),
                TextColumn::make('user.name')
                    ->label(__('messages.t_earning_table.user_name'))
                    ->searchable(),
                TextColumn::make('order.subtotal_amount')
                    ->label(__('messages.t_earning_table.amount'))
                    ->sortable()
                    ->searchable()
                    ->formatStateUsing(fn($state) => formatPriceWithCurrency($state)),
                SpatieMediaLibraryImageColumn::make('proof_attachment')
                    ->collection('order_refund_proof')
                    ->label(fn() => $this->activeTab === __('messages.t_ap_buyer')
                        ? __('messages.t_ap_payment_proof')
                        : __('messages.t_ap_delivery_proof'))
                    ->size(50)
                    ->square()
                    ->visibility('public')
                    ->extraImgAttributes(fn($record): array => [
                        'alt' => $this->activeTab === __('messages.t_ap_buyer')
                            ? __('messages.t_ap_payment_proof')
                            : __('messages.t_ap_delivery_proof'),
                    ])
                    ->action(
                        Action::make('viewProof')
                            ->modalHeading(fn() => $this->activeTab === __('messages.t_ap_buyer')
                                ? __('messages.t_ap_payment_proof')
                                : __('messages.t_ap_delivery_proof'))
                            ->modalContent(fn($record) => view('filament.modals.view-proof', [
                                'image' => $record->getFirstMediaUrl('order_refund_proof')
                            ]))
                            ->modalSubmitAction(false) // Hides the Submit button
                            ->modalCancelAction(false) // Hides the Cancel button
                            ->modalWidth('lg') // Optional: Adjust modal width
                    ),
                SpatieMediaLibraryImageColumn::make('summary_attachment')
                    ->collection('order_summary_proof')
                    ->visibility('public')
                    ->label(__('messages.t_ap_refund_order_summary_proof_images'))
                    ->size(50)
                    ->square()
                    ->extraImgAttributes(fn($record): array => [
                        'alt' => __('messages.t_ap_refund_order_summary_proof_images'),
                    ])
                    ->action(
                        Action::make('viewSummaryProof')
                            ->modalHeading(__('messages.t_ap_refund_order_summary_proof_images'))
                            ->modalContent(fn($record) => view('filament.modals.view-proof', [
                                'image' => $record->getFirstMediaUrl('order_summary_proof')
                            ]))
                            ->modalSubmitAction(false) // Hides the Submit button
                            ->modalCancelAction(false) // Hides the Cancel button
                            ->modalWidth('lg') // Optional: Adjust modal width
                    ),
                TextColumn::make('order.payment_method')
                    ->label(__('messages.t_ap_pay_type'))
                    ->searchable()
                    ->visible(fn($record) => $this->activeTab == __('messages.t_ap_buyer')),
                TextColumn::make('order.transaction_id')
                    ->label(__('messages.t_ad_wallet_transaction_id'))
                    ->searchable()
                    ->visible(fn($record) => $this->activeTab == __('messages.t_ap_buyer')),
                TextColumn::make('status')
                    ->label(__('messages.t_ap_status'))
                    ->badge()
                    ->sortable()
                    ->searchable()
                    ->formatStateUsing(fn($state) => match ($state) {
                        'pending' => 'Pending',
                        'approved' => 'Approved',
                        'rejected' => 'Rejected',
                    })
                    ->color(fn($state) => match ($state) {
                        'pending' => 'warning',
                        'approved' => 'success',
                        'rejected' => 'danger',
                    }),
            ])
            ->filters([
                //
            ])
            ->actions([
                Action::make('order_approve')
                    ->action(function ($record) {

                        $record->update(['status' => 'approved']);

                        $this->UpdateStatusToRefund($record);
                    })
                    ->button()
                    ->color('success')
                    ->visible(function ($record) {
                        return $record->where('status', 'pending');
                    })
                    ->label(__('messages.t_ap_approve'))
                    ->requiresConfirmation()
                    ->modalHeading(__('messages.t_ap_approve_refund'))
                    ->modalDescription(__('messages.t_ap_approve_refund_description'))
                    ->successNotificationTitle(__('messages.t_ap_refund_approval_notification')),
                Action::make('order_reject')
                    ->action(function ($record) {
                        $record->update(['status' => 'rejected']);

                        $order = $record->order;
                        $order->histories()->where('action', 'order_rejected')->first()?->update([
                            'action_date' => now()
                        ]);
                    })
                    ->button()
                    ->color('danger')
                    ->visible(function ($record) {
                        return $record->where('status', 'pending');
                    })
                    ->label(__('messages.t_ap_reject'))
                    ->requiresConfirmation()
                    ->modalHeading(__('messages.t_ap_reject_refund'))
                    ->modalDescription(__('messages.t_ap_reject_refund_description'))
                    ->successNotificationTitle(__('messages.t_ap_refund_reject_notification')),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public function UpdateStatusToRefund($record)
    {
        $order = $record->order;

        $order->histories()->where('action', 'order_rejected')->first()?->update([
            'action_date' => now()
        ]);

        if ($record->type == 'buyer') {

            $order->where('id', $record->order_id)->first()?->update([
                'order_status' => 'reviewed'
            ]);

        } else {

            if (isWalletSystemPluginEnabled()) {

                $wallet = Wallet::firstOrCreate(
                    ['user_id' => $record->user_id],  // Query condition
                    ['balance' => 0]              // Default balance for new wallet
                );

                $wallet->increment('balance', $order->subtotal_amount);

                // Create a transaction record for the wallet
                $transaction = $wallet->transactions()->create([
                    'user_id' => $record->user_id,
                    'amount' => $order->subtotal_amount,
                    'transaction_reference' => $order->order_number,
                    'status' => 'refund',
                ]);

                $transaction->payable()->associate($order)->save();

                $order->where('id', $record->order_id)->first()?->update([
                    'order_status' => 'reviewed'
                ]);
            }

        }
    }
}
